import {
    LeaderboardConstants,
} from "./Interfaces/LeaderboardConstants";

export class FullLeaderboardAnimations {
    private animationTween: Tween<any> = null;
    private isScreenTextureInPreview: boolean = null;

    constructor(private mainScreenTransform: ScreenTransform,
                private previewScreenTransform: ScreenTransform,
                private topBorderScreenTransform: ScreenTransform,
                private roundedCornersTransform: Transform,
                private previewParentScreenTransform: ScreenTransform) {

    }

    setIsScreenTextureInPreview(isScreenTex: boolean): void {
        this.isScreenTextureInPreview = isScreenTex;
    }

    isPlaying(): boolean {
        if (isNull(this.animationTween)) {
            return false;
        }
        if (isNull(this.animationTween.isPlaying)) {
            return false;
        }
        return this.animationTween.isPlaying();
    }

    forceStopAnimationIfPlaying() {
        if (!isNull(this.animationTween)) {
            try {
                this.animationTween.stop();
                TWEEN.remove(this.animationTween);
            } catch (e) {}
        }
    }

    playAnimation(toShow: boolean, onComplete: () => void): void {
        const startPos = this.mainScreenTransform.anchors.getCenter();
        const endPos = toShow ? new vec2(0, 0) : new vec2(0, -2);
        const duration = startPos.distance(endPos) * LeaderboardConstants.FullLeaderboard.ANIMATION_DURATION_MS / 2;

        if (duration === 0) {
            return;
        }

        this.forceStopAnimationIfPlaying();

        const startRect = this.previewScreenTransform.anchors;
        const endRect = toShow ? LeaderboardConstants.FullLeaderboard.CUSTOM_TEXTURE_END_RECT
            : LeaderboardConstants.FullLeaderboard.CUSTOM_TEXTURE_START_RECT;

        const parentStartRect = this.previewParentScreenTransform.anchors;
        const parentEndRect = toShow ? LeaderboardConstants.FullLeaderboard.CUSTOM_TEXTURE_PARENT_END_RECT
            : LeaderboardConstants.FullLeaderboard.CUSTOM_TEXTURE_PARENT_START_RECT;

        const topBorderBottomStartValue = this.topBorderScreenTransform.anchors.bottom;
        const topBorderBottomEndValue = toShow ? LeaderboardConstants.FullLeaderboard.TOP_BORDER_BOTTOM_END_VALUE
            : LeaderboardConstants.FullLeaderboard.TOP_BORDER_BOTTOM_START_VALUE;

        const roundCornersScaleStartValue = this.roundedCornersTransform.getLocalScale().x;
        const roundCornersScaleEndValue = toShow ? LeaderboardConstants.FullLeaderboard.ROUND_CORNER_SCALE_X_END_VALUE
            : LeaderboardConstants.FullLeaderboard.ROUND_CORNER_SCALE_X_START_VALUE;

        const from = {
            y: startPos.y,
            topBorderBottom: topBorderBottomStartValue,
            roundCornersScale: roundCornersScaleStartValue };
        const to = {
            y: endPos.y,
            topBorderBottom: topBorderBottomEndValue,
            roundCornersScale: roundCornersScaleEndValue };

        this.enrichObjectWithRect(from, startRect, "customTexture");
        this.enrichObjectWithRect(to, endRect, "customTexture");

        this.enrichObjectWithRect(from, parentStartRect, "customTextureParent");
        this.enrichObjectWithRect(to, parentEndRect, "customTextureParent");

        this.animationTween = new TWEEN.Tween(from)
            .to(to, duration)
            .onUpdate((obj) => {
                this.mainScreenTransform.anchors.setCenter(new vec2(0, obj.y));
                if (this.isScreenTextureInPreview) {
                    this.previewScreenTransform.anchors = this.constructRectWithPrefix(obj, "customTexture");
                    this.previewParentScreenTransform.anchors = this.constructRectWithPrefix(obj, "customTextureParent");
                }

                this.topBorderScreenTransform.anchors.bottom = obj.topBorderBottom;
                this.roundedCornersTransform.setLocalScale(new vec3(obj.roundCornersScale, 1, 1));

            })
            .easing(TWEEN.Easing.Quadratic.InOut)
            .onComplete(onComplete)
            .start();
    }

    private enrichObjectWithRect(obj: Object, rect: Rect, prefix: string): void {
        obj[prefix + "left"] = rect.left;
        obj[prefix + "right"] = rect.right;
        obj[prefix + "bottom"] = rect.bottom;
        obj[prefix + "top"] = rect.top;
    }

    private constructRectWithPrefix(obj: Object, prefix: string): Rect {
        return Rect.create(obj[prefix + "left"], obj[prefix + "right"], obj[prefix + "bottom"], obj[prefix + "top"]);
    }
}
